<?php
require_once 'includes/db.php';
require_once 'includes/mailer.php'; // ADD THIS LINE

// 1. If user is already logged in, redirect them to their profile
if (isset($_SESSION['user_id'])) {
    header("Location: user-profile.php");
    exit;
}

$error = '';

// 2. Handle Form Submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $full_name = sanitize_input($_POST['full_name']);
    $phone     = sanitize_input($_POST['phone_number']);
    $email     = sanitize_input($_POST['email']);
    $pass      = $_POST['password'];
    $confirm   = $_POST['password_confirm'];

    // Basic Validation
    if ($pass !== $confirm) {
        $error = "Passwords do not match.";
    } else {
        // Check if email already exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $res = $stmt->get_result();

        if ($res->num_rows > 0) {
            $error = "This email is already registered. Please login.";
        } else {
            // Hash Password
            $hashed_pass = password_hash($pass, PASSWORD_BCRYPT);

            // Insert into Database
            $insert = $conn->prepare("INSERT INTO users (full_name, email, password, phone, created_at) VALUES (?, ?, ?, ?, NOW())");
            $insert->bind_param("ssss", $full_name, $email, $hashed_pass, $phone);

            if ($insert->execute()) {
                // Success: Create User Session
                $_SESSION['user_id'] = $insert->insert_id;
                $_SESSION['user_name'] = $full_name;
                $_SESSION['user_email'] = $email;

                // --- NEW: Set a flag to send the email on the next page load ---
                $_SESSION['trigger_welcome_email'] = true;

                // Redirect to Profile Instantly
                header("Location: user-profile.php");
                exit;
            } else {
                $error = "Something went wrong. Please try again.";
            }
        }
        $stmt->close();
    }
}

$page_title = "Create an Account";
include 'includes/header.php';
?>

<!-- Start Page Title -->
<div class="page-title-area">
    <div class="container">
        <div class="page-title-content">
            <h2>Create an Account</h2>
            <ul>
                <li><a href="index.php">Home</a></li>
                <li>Signup</li>
            </ul>
        </div>
    </div>
</div>
<!-- End Page Title -->

<!-- Start SignUP Area -->
<section class="signup-area pt-100">
    <div class="container">
        <div class="signup-content">
            <h2>Create an Account</h2>

            <!-- Display Errors if any -->
            <?php if ($error): ?>
                <div class="alert alert-danger"><?= $error ?></div>
            <?php endif; ?>

            <form class="signup-form" method="POST" action="signup.php">
                <div class="form-group">
                    <label>Full Name</label>
                    <input type="text" class="form-control" placeholder="Enter your full name" name="full_name" required>
                </div>

                <div class="form-group">
                    <label>Phone Number</label>
                    <input type="text" class="form-control" placeholder="Enter your phone number" name="phone_number" required>
                </div>

                <div class="form-group">
                    <label>Email</label>
                    <input type="email" class="form-control" placeholder="Enter your email address" name="email" required>
                </div>

                <!-- Password Field -->
                <div class="form-group">
                    <label>Password</label>
                    <div class="password-wrapper">
                        <input type="password" class="form-control" placeholder="Enter your password" name="password" required>
                        <i class='bx bx-hide toggle-password'></i>
                    </div>
                </div>

                <!-- Confirm Password Field -->
                <div class="form-group">
                    <label>Confirm Password</label>
                    <div class="password-wrapper">
                        <input type="password" class="form-control" placeholder="Confirm your password" name="password_confirm" required>
                        <i class='bx bx-hide toggle-password'></i>
                    </div>
                </div>

                <!-- Removed window.location.href - handled by PHP redirect now -->
                <button type="submit" class="default-btn">Create Account</button>

                <div class="text-center mt-3">
                    <p>Already have an account? <a href="login.php">Login here</a></p>
                </div>

                <a href="index.php" class="return-store">or Return to Store</a>
            </form>
        </div>
    </div>
</section>

<?php include 'includes/footer.php'; ?>