<?php
require_once 'includes/db.php';

// --- 1. VALIDATE TOKEN ---
$token = $_GET['token'] ?? '';
$error = '';
$show_form = false;

if (empty($token)) {
    $error = "Invalid reset link. Please request a new one.";
} else {
    // Find user by token
    $stmt = $conn->prepare("SELECT id, token_expires_at FROM users WHERE reset_token = ?");
    $stmt->bind_param("s", $token);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 1) {
        $user = $result->fetch_assoc();

        // Check if token has expired
        if (strtotime($user['token_expires_at']) < time()) {
            $error = "This password reset link has expired. Please request a new one.";
        } else {
            // Token is valid, show the form
            $show_form = true;
        }
    } else {
        $error = "Invalid or expired reset link.";
    }
}

// --- 2. HANDLE NEW PASSWORD SUBMISSION ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reset_password'])) {
    $new_pass = $_POST['new_password'];
    $confirm_pass = $_POST['confirm_password'];
    $user_token = $_POST['token']; // Get token from hidden field

    if ($new_pass !== $confirm_pass) {
        $error = "Passwords do not match. Please try again.";
        $show_form = true; // Re-show the form
        $token = $user_token; // Keep token available for the form
    } else {
        // Find user ID again to be safe
        $stmt = $conn->prepare("SELECT id FROM users WHERE reset_token = ?");
        $stmt->bind_param("s", $user_token);
        $stmt->execute();
        $user_id = $stmt->get_result()->fetch_assoc()['id'];

        // Update password and clear the token
        $hashed_pass = password_hash($new_pass, PASSWORD_BCRYPT);
        $update_stmt = $conn->prepare("UPDATE users SET password = ?, reset_token = NULL, token_expires_at = NULL WHERE id = ?");
        $update_stmt->bind_param("si", $hashed_pass, $user_id);

        if ($update_stmt->execute()) {
            // Success! Redirect to login with a success message
            $_SESSION['flash_msg'] = ["type" => "success", "text" => "Your password has been updated! You can now log in."];
            header("Location: login.php");
            exit;
        } else {
            $error = "Something went wrong. Please try again.";
        }
    }
}

$page_title = "Reset Your Password";
include 'includes/header.php';
?>

<!-- Start Page Title -->
<div class="page-title-area">
    <div class="container">
        <div class="page-title-content">
            <h2>Reset Password</h2>
            <ul>
                <li><a href="index.php">Home</a></li>
                <li>Reset Password</li>
            </ul>
        </div>
    </div>
</div>
<!-- End Page Title -->

<!-- Start Reset Password Area -->
<section class="login-area pt-100">
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-lg-6 col-md-8">
                <div class="login-content">
                    <h2>Create a New Password</h2>
                    <p class="text-muted mb-4">Please enter and confirm your new password below.</p>

                    <!-- Display Errors -->
                    <?php if ($error): ?>
                        <div class="alert alert-danger"><?= $error ?></div>
                    <?php endif; ?>

                    <?php if ($show_form): ?>
                        <form class="login-form" method="POST" action="reset-password.php">
                            <!-- Hidden token field to pass it along -->
                            <input type="hidden" name="token" value="<?= htmlspecialchars($token) ?>">

                            <!-- New Password Field -->
                            <div class="form-group">
                                <label>New Password</label>
                                <div class="password-wrapper">
                                    <input type="password" name="new_password" class="form-control" placeholder="Enter your new password" required>
                                    <i class='bx bx-hide toggle-password'></i>
                                </div>
                            </div>

                            <!-- Confirm New Password Field -->
                            <div class="form-group">
                                <label>Confirm New Password</label>
                                <div class="password-wrapper">
                                    <input type="password" name="confirm_password" class="form-control" placeholder="Confirm your new password" required>
                                    <i class='bx bx-hide toggle-password'></i>
                                </div>
                            </div>

                            <button type="submit" name="reset_password" class="default-btn">Update Password</button>
                        </form>
                    <?php else: ?>
                        <!-- If form is not shown due to error, give a link back -->
                        <div class="text-center mt-4">
                            <a href="forgot-password.php" class="default-btn">Request a New Link</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</section>
<!-- End Reset Password Area -->

<?php include 'includes/footer.php'; ?>