<?php
require_once 'includes/db.php';

// --- 1. SECURITY: Authentication ---
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php?redirect=profile");
    exit;
}
$user_id = $_SESSION['user_id'];

// --- 2. SECURITY: Authorization & Data Fetching ---
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header("Location: user-profile.php?active=orders");
    exit;
}
$order_id = (int)$_GET['id'];

// Fetch the order ONLY if the ID and USER_ID match
$stmt_order = $conn->prepare("SELECT * FROM orders WHERE id = ? AND user_id = ?");
$stmt_order->bind_param("ii", $order_id, $user_id);
$stmt_order->execute();
$order_res = $stmt_order->get_result();

if ($order_res->num_rows === 0) {
    header("Location: user-profile.php?active=orders");
    exit;
}

$order = $order_res->fetch_assoc();
$shipping = json_decode($order['shipping_address_details'], true);

// --- 3. FETCH ORDER ITEMS ---
$items_sql = "SELECT oi.*, p.name, p.slug,
              (SELECT image_path FROM product_images pi 
               WHERE pi.product_id = oi.product_id 
               AND (pi.color_group = oi.color OR pi.color_group = 'Standard')
               ORDER BY pi.is_primary DESC LIMIT 1) as image_path
              FROM order_items oi
              JOIN products p ON oi.product_id = p.id
              WHERE oi.order_id = ?";
$stmt_items = $conn->prepare($items_sql);
$stmt_items->bind_param("i", $order_id);
$stmt_items->execute();
$items_res = $stmt_items->get_result();

$order_items = [];
while ($item = $items_res->fetch_assoc()) {
    $order_items[] = $item;
}

include 'includes/header.php';
?>

<!-- Start Page Title -->
<div class="page-title-area">
    <div class="container">
        <div class="page-title-content">
            <h2>Order Details</h2>
            <ul>
                <li><a href="index.php">Home</a></li>
                <li><a href="user-profile.php?active=orders">My Orders</a></li>
                <li>#<?= str_pad($order['id'], 6, '0', STR_PAD_LEFT) ?></li>
            </ul>
        </div>
    </div>
</div>

<section class="ptb-100" style="background-color: #f9f9f9;">
    <div class="container">
        <div class="row">

            <!-- LEFT COLUMN: Items & Info -->
            <div class="col-lg-8">

                <!-- 1. Order Header & Items -->
                <div class="order-view-card">
                    <div class="order-view-header">
                        <div>
                            <h4>Order #<?= str_pad($order['id'], 6, '0', STR_PAD_LEFT) ?></h4>
                            <small class="text-muted">Placed on <?= date('F d, Y \a\t h:i A', strtotime($order['created_at'])) ?></small>
                        </div>
                        <span class="status-badge status-<?= $order['order_status'] ?>">
                            <?= htmlspecialchars($order['order_status']) ?>
                        </span>
                    </div>
                    <div class="order-view-body">
                        <?php
                        $subtotal = 0;
                        foreach ($order_items as $item):
                            $line_total = $item['price'] * $item['quantity'];
                            $subtotal += $line_total;
                        ?>
                            <div class="item-row">
                                <a href="product-details.php?slug=<?= $item['slug'] ?>">
                                    <img src="<?= $item['image_path'] ?? 'assets/img/placeholder.png' ?>" class="item-img" alt="<?= htmlspecialchars($item['name']) ?>">
                                </a>
                                <div class="item-details flex-grow-1">
                                    <a href="product-details.php?slug=<?= $item['slug'] ?>" class="text-dark text-decoration-none">
                                        <h6><?= htmlspecialchars($item['name']) ?></h6>
                                    </a>
                                    <div class="item-meta">
                                        <span>Size: <strong><?= $item['size'] ?></strong></span> |
                                        <span>Color: <strong><?= $item['color'] ?></strong></span>
                                    </div>
                                    <div class="item-meta mt-1">
                                        Price: ₹<?= number_format($item['price'], 2) ?> x <?= $item['quantity'] ?>
                                    </div>
                                </div>
                                <div class="item-total font-weight-bold">
                                    <strong>₹<?= number_format($line_total, 2) ?></strong>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <!-- 2. Address & Payment Info -->
                <div class="row">
                    <div class="col-md-6 mb-4">
                        <div class="order-view-card h-100">
                            <div class="order-view-body h-100">
                                <div class="info-box">
                                    <h5><i class='bx bxs-truck'></i> Shipping Address</h5>
                                    <p class="font-weight-bold mb-1"><?= htmlspecialchars($shipping['full_name']) ?></p>
                                    <p>
                                        <?= htmlspecialchars($shipping['address_1']) ?><br>
                                        <?php if (!empty($shipping['address_2'])): ?>
                                            <?= htmlspecialchars($shipping['address_2']) ?><br>
                                        <?php endif; ?>
                                        <?= htmlspecialchars($shipping['city']) ?>, <?= htmlspecialchars($shipping['state']) ?> - <?= htmlspecialchars($shipping['pincode']) ?>
                                    </p>
                                    <p class="mt-2"><i class='bx bxs-phone'></i> <?= htmlspecialchars($shipping['phone']) ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 mb-4">
                        <div class="order-view-card h-100">
                            <div class="order-view-body h-100">
                                <div class="info-box">
                                    <h5><i class='bx bxs-credit-card'></i> Payment Info</h5>
                                    <p><strong>Method:</strong> <?= strtoupper($order['payment_method']) ?></p>

                                    <!-- NEW: Display Razorpay Payment ID if it exists -->
                                    <?php if (!empty($order['razorpay_payment_id'])): ?>
                                        <p><strong>Payment ID:</strong> <code class="text-muted" style="font-size: 13px;"><?= htmlspecialchars($order['razorpay_payment_id']) ?></code></p>
                                    <?php endif; ?>

                                    <p><strong>Status:</strong>
                                        <span class="<?= $order['payment_status'] == 'Completed' ? 'text-success' : 'text-warning' ?> font-weight-bold">
                                            <?= htmlspecialchars($order['payment_status']) ?>
                                        </span>
                                    </p>

                                    <?php if (!empty($order['tracking_id'])): ?>
                                        <div class="mt-3 p-2 bg-white border rounded">
                                            <small class="text-muted d-block">Tracking Number:</small>
                                            <strong><?= htmlspecialchars($order['tracking_id']) ?></strong>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

            </div>

            <!-- RIGHT COLUMN: Summary -->
            <div class="col-lg-4">
                <div class="order-view-card">
                    <div class="order-view-header">
                        <h4>Order Summary</h4>
                    </div>
                    <div class="order-view-body">
                        <table class="summary-table">
                            <tr>
                                <td>Item Subtotal</td>
                                <td class="text-end">₹<?= number_format($subtotal, 2) ?></td>
                            </tr>
                            <?php
                            // Calculate discount from order total logic
                            $discount = $order['discount_amount'] ?? ($subtotal - $order['total_amount']);
                            if ($discount > 0.01):
                            ?>
                                <tr>
                                    <td>Discount <?php if (!empty($order['coupon_used'])) echo '<small class="text-muted">(' . $order['coupon_used'] . ')</small>'; ?></td>
                                    <td class="text-end text-danger">-₹<?= number_format($discount, 2) ?></td>
                                </tr>
                            <?php endif; ?>
                            <tr>
                                <td>Shipping</td>
                                <td class="text-end text-success">Free</td>
                            </tr>
                            <tr>
                                <td><strong>Total Amount</strong></td>
                                <td class="text-end">₹<?= number_format($order['total_amount'], 2) ?></td>
                            </tr>
                        </table>

                        <div class="mt-4">
                            <a href="contact.php" class="optional-btn w-100 text-center mb-2">Need Help?</a>
                            <a href="user-profile.php?active=orders" class="default-btn w-100 text-center"><i class='bx bx-arrow-back'></i> Back to Orders</a>
                        </div>
                    </div>
                </div>
            </div>

        </div>
    </div>
</section>

<?php include 'includes/footer.php'; ?>