<?php
// apply-coupon.php
require_once 'includes/db.php';
header('Content-Type: application/json');

// NEW: Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode([
        'success' => false, 
        'message' => '🔒 Please login to your account to use coupon codes.'
    ]);
    exit;
}

// NEW: Handle Coupon Removal
if (isset($_POST['action']) && $_POST['action'] === 'remove') {
    unset($_SESSION['applied_coupon']);
    echo json_encode(['success' => true, 'message' => 'Coupon removed']);
    exit;
}


if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request']);
    exit;
}

$code = isset($_POST['code']) ? strtoupper(trim(sanitize_input($_POST['code']))) : '';
// Use checkout items if they exist (Buy It Now), otherwise fallback to main cart
$cart = !empty($_SESSION['checkout_items']) ? $_SESSION['checkout_items'] : ($_SESSION['cart'] ?? []);

if (empty($code)) {
    echo json_encode(['success' => false, 'message' => 'Please enter a coupon code']);
    exit;
}

if (empty($cart)) {
    echo json_encode(['success' => false, 'message' => 'Cart is empty']);
    exit;
}

// 1. Fetch Coupon Details
$stmt = $conn->prepare("SELECT * FROM coupons WHERE code = ? AND is_active = 1");
$stmt->bind_param("s", $code);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    echo json_encode(['success' => false, 'message' => 'Invalid or inactive coupon code']);
    exit;
}

$coupon = $result->fetch_assoc();

// 2. Check Expiry
if ($coupon['expiry_date'] && strtotime($coupon['expiry_date']) < time()) {
    echo json_encode(['success' => false, 'message' => 'This coupon has expired']);
    exit;
}

// 3. Check Global Usage Limit
if ($coupon['usage_limit'] > 0 && $coupon['used_count'] >= $coupon['usage_limit']) {
    echo json_encode(['success' => false, 'message' => 'This coupon has reached its usage limit']);
    exit;
}

// 4. Check Per-User Limit (If logged in)
$user_id = $_SESSION['user_id'];
if ($coupon['limit_per_user'] > 0) {
    $stmt_usage = $conn->prepare("SELECT COUNT(*) as user_usage FROM orders WHERE user_id = ? AND coupon_used = ? AND order_status != 'Cancelled'");
    $stmt_usage->bind_param("is", $user_id, $code);
    $stmt_usage->execute();
    $usage_result = $stmt_usage->get_result()->fetch_assoc();
    
    if ($usage_result['user_usage'] >= $coupon['limit_per_user']) {
        echo json_encode(['success' => false, 'message' => 'You have already used this coupon the maximum number of times.']);
        exit;
    }
}


// 5. Calculate Applicable Subtotal & Validate Cart Contents
$applicable_subtotal = 0;
$cart_total = 0;
$has_ineligible_items = false;

foreach ($cart as $item) {
    $line_total = $item['price'] * $item['quantity'];
    $cart_total += $line_total;

    if ($coupon['category_id'] == 0) { // Global coupon
        $applicable_subtotal += $line_total;
    } else { // Category-specific coupon
        $p_id = $item['product_id'];
        $cat_check = $conn->query("SELECT category_id FROM products WHERE id = $p_id");
        $prod_data = $cat_check->fetch_assoc();
        
        if ($prod_data['category_id'] == $coupon['category_id']) {
            $applicable_subtotal += $line_total;
        } else {
            // Found an item not in the coupon's category
            $has_ineligible_items = true;
        }
    }
}

// *** NEW VALIDATION STEP ***
// If the coupon is for a specific category AND there are other items in the cart, fail.
if ($coupon['category_id'] > 0 && $has_ineligible_items) {
    echo json_encode([
        'success' => false, 
        'message' => 'This coupon is only valid for a specific category and cannot be used with other items in your cart.'
    ]);
    exit;
}

// 6. Check Minimum Spend
if ($coupon['min_spend'] > 0 && $cart_total < $coupon['min_spend']) {
    echo json_encode([
        'success' => false, 
        'message' => 'Minimum spend of ₹' . number_format($coupon['min_spend']) . ' required'
    ]);
    exit;
}

if ($applicable_subtotal == 0) {
    echo json_encode([
        'success' => false, 
        'message' => 'This coupon is not applicable to items in your cart'
    ]);
    exit;
}

// 7. Calculate Discount
$discount = 0;
if ($coupon['discount_type'] == 'percentage') {
    $discount = ($applicable_subtotal * $coupon['discount_value']) / 100;
} else {
    $discount = $coupon['discount_value'];
}

// Ensure discount doesn't exceed total
if ($discount > $cart_total) {
    $discount = $cart_total;
}

$final_total = $cart_total - $discount;

// 8. Save to Session (so it persists on refresh)
$_SESSION['applied_coupon'] = [
    'code' => $code,
    'discount' => $discount,
    'id' => $coupon['id']
];

// NEW: Save message for the UI to show after reload
$_SESSION['coupon_success_msg'] = 'Coupon applied successfully!';

echo json_encode([
    'success' => true,
    'message' => 'Coupon applied successfully!',
    'discount' => $discount,
    'new_total' => $final_total
]);
exit;