<?php
require_once 'includes/db_config.php';
check_login();

// Redirect if no ID provided
if (!isset($_GET['id']) || !($product_id = (int)$_GET['id'])) {
    header("Location: products.php");
    exit;
}

// --- 1. HANDLE UPDATE SUBMISSION ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_product'])) {

    // A. General Info
    $name = sanitize_input($_POST['name']);
    $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $name)));
    $short = sanitize_input($_POST['short_description']);
    $desc = $_POST['description']; // HTML allowed
    $cat_id = (int)$_POST['category_id'];
    $price = (float)$_POST['price'];
    $sale = !empty($_POST['sale_price']) ? (float)$_POST['sale_price'] : NULL;
    $sku = sanitize_input($_POST['sku']);
    $active = isset($_POST['is_active']) ? 1 : 0;

    // Capture default color
    $default_color = sanitize_input($_POST['default_color_group'] ?? 'Standard');

    // B. Specifications (JSON)
    $specs = [];
    if (isset($_POST['spec_key'])) {
        for ($i = 0; $i < count($_POST['spec_key']); $i++) {
            if (!empty($_POST['spec_key'][$i])) {
                $specs[$_POST['spec_key'][$i]] = $_POST['spec_val'][$i];
            }
        }
    }
    $info = json_encode($specs);

    // C. Update Main Product Table
    $stmt = $conn->prepare("UPDATE products SET name=?, slug=?, sku=?, category_id=?, short_description=?, description=?, additional_info=?, price=?, sale_price=?, is_active=?, default_color_group=? WHERE id=?");
    $stmt->bind_param("sssisssddisi", $name, $slug, $sku, $cat_id, $short, $desc, $info, $price, $sale, $active, $default_color, $product_id);
    $stmt->execute();

    // D. VARIATIONS LOGIC

    // 1. Wipe and Re-insert SIZES
    $conn->query("DELETE FROM product_sizes WHERE product_id = $product_id");

    $globalHasColor = $_POST['has_color_option'] ?? '0';

    // --- INDUSTRY STANDARD FIX: LABEL RESET ---
    // If the user switched from "No" to "Yes", they are turning 'Standard' images into a specific color.
    // If they switched from "Yes" to "No", they are turning colored images into 'Standard'.
    // To prevent "Ghost Groups", we identify the colors being used in this submission.
    $submittedColors = [];
    if (isset($_POST['variants'])) {
        foreach ($_POST['variants'] as $v) {
            $c = ($globalHasColor == '1') ? sanitize_input($v['color']) : 'Standard';
            if (!empty($c)) $submittedColors[] = $c;
        }
    }

    if (isset($_POST['variants'])) {
        foreach ($_POST['variants'] as $idx => $var) {

            $colorName = ($globalHasColor == '1') ? sanitize_input($var['color']) : 'Standard';

            // 1. INSERT SIZES
            if (isset($var['sizes']['name'])) {
                foreach ($var['sizes']['name'] as $si => $sName) {
                    $sQty = (int)$var['sizes']['qty'][$si];
                    if (!empty($sName)) {
                        $stmt = $conn->prepare("INSERT INTO product_sizes (product_id, color_group, size, stock_quantity) VALUES (?, ?, ?, ?)");
                        $stmt->bind_param("issi", $product_id, $colorName, $sName, $sQty);
                        $stmt->execute();
                    }
                }
            }

            // 2. HANDLE IMAGES with proper primary image per color
            $selected_primary_val = $var['primary_img'] ?? null;

            // First, reset ALL images for this color group to non-primary
            $conn->query("UPDATE product_images SET is_primary = 0 WHERE product_id = $product_id AND color_group = '$colorName'");

            // Handle primary selection
            if ($selected_primary_val && strpos($selected_primary_val, 'new_') === 0) {
                // New upload selected as primary - we'll handle this after upload
                $new_primary_index = (int)str_replace('new_', '', $selected_primary_val);
            } elseif (is_numeric($selected_primary_val)) {
                // Existing image selected as primary
                $existing_primary_id = (int)$selected_primary_val;

                // Set this specific image as primary AND ensure it's in correct color group
                $conn->query("UPDATE product_images SET is_primary = 1, color_group = '$colorName' WHERE id = $existing_primary_id");
            } else {
                // No primary selected - set first image of this color as primary
                $conn->query("UPDATE product_images SET is_primary = 1 
                  WHERE product_id = $product_id 
                  AND color_group = '$colorName' 
                  ORDER BY id ASC LIMIT 1");
            }

            // --- CRITICAL ADDITION: ORPHAN IMAGE MIGRATION ---
            // If this is the FIRST variant group, and we are in "No Color" mode, 
            // OR if this is the ONLY variant group, we grab any images that don't 
            // belong to any of the current submitted colors and force them into this group.
            if ($idx == 0) {
                $quotedColors = "'" . implode("','", $submittedColors) . "'";
                $conn->query("UPDATE product_images SET color_group = '$colorName' WHERE product_id = $product_id AND color_group NOT IN ($quotedColors)");
            }

            // 3. PROCESS NEW UPLOADS with primary flag
            if (isset($_FILES['variants']['name'][$idx]['images'])) {
                $fileAry = $_FILES['variants'];
                $count = count($fileAry['name'][$idx]['images']);

                for ($k = 0; $k < $count; $k++) {
                    if ($fileAry['error'][$idx]['images'][$k] === 0) {
                        $tmpName = $fileAry['tmp_name'][$idx]['images'][$k];
                        $ext = strtolower(pathinfo($fileAry['name'][$idx]['images'][$k], PATHINFO_EXTENSION));
                        $newName = $slug . '_' . $colorName . '_' . uniqid() . '.' . $ext;
                        $target = '../assets/uploads/products/' . $newName;

                        if (move_uploaded_file($tmpName, $target)) {
                            $dbPath = 'assets/uploads/products/' . $newName;

                            // Check if this new upload is the selected primary
                            $isPrim = 0;
                            if (isset($new_primary_index) && $k === $new_primary_index) {
                                $isPrim = 1;
                            } elseif (!isset($new_primary_index) && $k === 0) {
                                // If no primary was selected from existing images, first new upload becomes primary
                                $check_existing = $conn->query("SELECT COUNT(*) as count FROM product_images WHERE product_id = $product_id AND color_group = '$colorName' AND is_primary = 1");
                                $row = $check_existing->fetch_assoc();
                                if ($row['count'] == 0) {
                                    $isPrim = 1;
                                }
                            }

                            $stmt = $conn->prepare("INSERT INTO product_images (product_id, image_path, color_group, is_primary) VALUES (?, ?, ?, ?)");
                            $stmt->bind_param("issi", $product_id, $dbPath, $colorName, $isPrim);
                            $stmt->execute();
                        }
                    }
                }
            }
        }
    }

    // E. Handle Explicit Deletions (Hidden inputs from JS for existing images)
    if (isset($_POST['delete_image_ids'])) {
        foreach ($_POST['delete_image_ids'] as $delID) {
            $did = (int)$delID;
            $res = $conn->query("SELECT image_path FROM product_images WHERE id = $did");
            if ($r = $res->fetch_assoc()) {
                if (file_exists('../' . $r['image_path'])) {
                    @unlink('../' . $r['image_path']);
                }
            }
            $conn->query("DELETE FROM product_images WHERE id = $did");
        }
    }

    $_SESSION['flash_message'] = '<div class="alert alert-success">Product updated successfully!</div>';
    header("Location: products.php");
    exit;
}

// --- 2. FETCH DATA FOR VIEW ---
$prod = $conn->query("SELECT * FROM products WHERE id = $product_id")->fetch_assoc();
if (!$prod) {
    header("Location: products.php");
    exit;
}

// Fetch categories (excluding virtual ones like New Arrivals/Best Sellers/On Sale)
$cats = $conn->query("SELECT id, name FROM categories WHERE slug NOT IN ('new-arrivals', 'best-sellers', 'on-sale') ORDER BY name ASC");
$specs = json_decode($prod['additional_info'] ?? '[]', true);

// Fetch Sizes & Images grouped by Color
$size_res = $conn->query("SELECT * FROM product_sizes WHERE product_id = $product_id");
$grouped_sizes = [];
while ($s = $size_res->fetch_assoc()) {
    $color = $s['color_group'] ?: 'Standard';
    $grouped_sizes[$color][] = $s;
}

$img_res = $conn->query("SELECT * FROM product_images WHERE product_id = $product_id ORDER BY id ASC");
$grouped_images = [];
while ($img = $img_res->fetch_assoc()) {
    $color = $img['color_group'] ?: 'Standard';
    $grouped_images[$color][] = $img;
}

// Build JS initialization object
$js_variants = [];
$all_colors_found = array_unique(array_merge(array_keys($grouped_sizes), array_keys($grouped_images)));

foreach ($all_colors_found as $cName) {
    $js_variants[] = [
        'color' => $cName,
        'sizes' => $grouped_sizes[$cName] ?? [],
        'images' => $grouped_images[$cName] ?? []
    ];
}

// Determine mode
$is_standard_mode = (count($js_variants) === 1 && $js_variants[0]['color'] === 'Standard');
$current_default_color = $prod['default_color_group'];

$page_title = "Edit Product";
include 'includes/admin_header.php';
?>

<div class="container-fluid">

    <!-- 1. BACK NAVIGATION -->
    <div class="page-header-nav">
        <a href="products.php"><i class="fas fa-arrow-left"></i></a>
        <h2>Edit Product: <?= htmlspecialchars($prod['name']) ?></h2>
    </div>

    <form id="productForm" method="POST" enctype="multipart/form-data">
        <!-- Hidden container for deletions -->
        <div id="delete-container"></div>

        <div class="row">
            <!-- 2. GENERAL INFO -->
            <div class="col-lg-12">
                <div class="product-card h-100">
                    <h5 class="section-title">1. General Information</h5>

                    <!-- Row 1 -->
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Product Name <span class="text-danger">*</span></label>
                                <input type="text" name="name" class="clean-input" value="<?= htmlspecialchars($prod['name']) ?>" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Category <span class="text-danger">*</span></label>
                                <select name="category_id" class="clean-input" required>
                                    <option value="">-- Select Category --</option>
                                    <?php $cats->data_seek(0);
                                    while ($c = $cats->fetch_assoc()): ?>
                                        <option value="<?= $c['id'] ?>" <?= ($c['id'] == $prod['category_id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($c['name']) ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Row 2 -->
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Short Description</label>
                                <textarea name="short_description" class="clean-input" rows="4"><?= htmlspecialchars($prod['short_description']) ?></textarea>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Detailed Description</label>
                                <textarea name="description" class="clean-input" rows="4"><?= htmlspecialchars($prod['description']) ?></textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Specs -->
                    <div class="mt-2 pt-3 border-top">
                        <label class="input-label mb-2">Additional Specifications</label>
                        <div id="specs-container"></div>
                        <button type="button" class="btn-soft mt-2" onclick="addSpecRow()">
                            <i class="fas fa-plus"></i> Add Specification
                        </button>
                    </div>
                </div>
            </div>

            <!-- 3. PRICING & ORG -->
            <div class="col-lg-12 mt-3">
                <div class="product-card h-100">
                    <h5 class="section-title">2. Pricing & Organization</h5>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Base Price (₹) <span class="text-danger">*</span></label>
                                <input type="number" name="price" class="clean-input" step="0.01" value="<?= $prod['price'] ?>" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Sale Price (₹)</label>
                                <input type="number" name="sale_price" class="clean-input" step="0.01" value="<?= $prod['sale_price'] ?>">
                            </div>
                        </div>
                    </div>
                    <hr>
                    <div class="row align-items-center">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">SKU</label>
                                <input type="text" name="sku" class="clean-input" value="<?= htmlspecialchars($prod['sku']) ?>">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="custom-control custom-switch mt-4">
                                <input type="checkbox" class="custom-control-input" id="isActive" name="is_active" <?= $prod['is_active'] ? 'checked' : '' ?>>
                                <label class="custom-control-label font-weight-bold" for="isActive">Active Status</label>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- 4. PRODUCT VARIATIONS -->
        <div class="row mt-3">
            <div class="col-12">
                <div class="product-card">
                    <h5 class="section-title">3. Product Variations</h5>

                    <!-- Color Question -->
                    <div class="bg-light p-3 rounded mb-4 border d-flex align-items-center gap-3">
                        <span class="font-weight-bold mr-3">Does this product have different colors?</span>
                        <div class="custom-control custom-radio custom-control-inline">
                            <input type="radio" id="colorYes" name="has_color_option" value="1" class="custom-control-input" onchange="toggleColorMode(true)">
                            <label class="custom-control-label" for="colorYes">Yes</label>
                        </div>
                        <div class="custom-control custom-radio custom-control-inline">
                            <input type="radio" id="colorNo" name="has_color_option" value="0" class="custom-control-input" onchange="toggleColorMode(false)">
                            <label class="custom-control-label" for="colorNo">No</label>
                        </div>
                    </div>

                    <!-- Color Definition Area -->
                    <div id="color-definition-area" style="display:none;" class="mb-4">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="pl-3 border-left">
                                    <label class="input-label">Step 1: Add Available Colors</label>
                                    <div class="d-flex gap-2 align-items-center mb-2">
                                        <select id="color-picker" class="clean-input" style="width: 200px;">
                                            <option value="">Select Color</option>

                                            <!-- Basics -->
                                            <option value="Red" data-hex="#da0103">Red</option>
                                            <option value="Blue" data-hex="#0000FF">Blue</option>
                                            <option value="Green" data-hex="#008000">Green</option>
                                            <option value="Black" data-hex="#000000">Black</option>
                                            <option value="White" data-hex="#FFFFFF">White</option>
                                            <option value="Grey" data-hex="#808080">Grey</option>

                                            <!-- Boutique / Premium Colors -->
                                            <option value="Maroon" data-hex="#800000">Maroon</option>
                                            <option value="Wine" data-hex="#722f37">Wine</option>
                                            <option value="Navy" data-hex="#000080">Navy</option>
                                            <option value="Teal" data-hex="#008080">Teal</option>
                                            <option value="Mustard" data-hex="#e1ad01">Mustard</option>
                                            <option value="Purple" data-hex="#800080">Purple</option>
                                            <option value="Brown" data-hex="#8b4513">Brown</option>

                                            <!-- Metallic & Luxury -->
                                            <option value="Gold" data-hex="#d4af37">Gold</option>
                                            <option value="Silver" data-hex="#c0c0c0">Silver</option>
                                            <option value="Ivory" data-hex="#fffff0">Ivory</option>
                                            <option value="Cream" data-hex="#fffdd0">Cream</option>
                                            <option value="Beige" data-hex="#f5f5dc">Beige</option>

                                            <!-- Pastels & Brights -->
                                            <option value="Peach" data-hex="#ffdab9">Peach</option>
                                            <option value="Pink" data-hex="#ffc0cb">Pink</option>
                                            <option value="Magenta" data-hex="#ff00ff">Magenta</option>
                                            <option value="Yellow" data-hex="#ffff00">Yellow</option>
                                            <option value="Orange" data-hex="#ffa500">Orange</option>
                                            <option value="Lavender" data-hex="#e6e6fa">Lavender</option>
                                            <option value="Turquoise" data-hex="#40e0d0">Turquoise</option>
                                        </select>
                                        <button type="button" class="btn btn-dark" onclick="addColorToMaster()">Add</button>
                                    </div>
                                    <div id="master-color-list" class="d-flex flex-wrap gap-2 mb-3"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group pt-3 pt-md-0">
                                    <label class="input-label">Step 2: Set Default Display Color</label>
                                    <select name="default_color_group" id="default-color-selector" class="clean-input" style="width: 250px;">
                                        <!-- Populated via JS -->
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Variants Container -->
                    <div id="variants-wrapper"></div>

                    <!-- Add Variant Button -->
                    <button type="button" class="btn-soft w-100 justify-content-center mt-3" id="add-variant-btn" onclick="addVariant()" style="display:none; border-style:dashed;">
                        <i class="fas fa-plus"></i> Add Another Color Variation
                    </button>
                </div>
            </div>
        </div>

        <!-- 5. SAVE BUTTON -->
        <div class="form-footer-action">
            <button type="submit" name="update_product" class="btn-save-product">Update Product</button>
        </div>

    </form>
</div>

<!-- JAVASCRIPT LOGIC -->
<script>
    // --- 1. INITIALIZATION DATA FROM PHP ---
    const dbVariants = <?= isset($js_variants) ? json_encode($js_variants) : '[]' ?>;
    const dbSpecs = <?= isset($specs) ? json_encode($specs) : '[]' ?>;
    const isStandardMode = <?= (isset($is_standard_mode) && $is_standard_mode) ? 'true' : 'false' ?>;
    const dbDefaultColor = "<?= $current_default_color ?? '' ?>";

    let variantCount = 0;
    let availableColors = [];
    let usedColors = [];

    // --- 2. STARTUP SEQUENCE ---
    document.addEventListener('DOMContentLoaded', function() {
        // Specs
        if (Object.keys(dbSpecs).length > 0) {
            for (const [key, value] of Object.entries(dbSpecs)) {
                addSpecRow(key, value);
            }
        } else {
            addSpecRow();
        }

        // Variants
        if (typeof dbVariants !== 'undefined' && dbVariants.length > 0) {
            if (isStandardMode) {
                // STANDARD MODE
                document.getElementById('colorNo').checked = true;

                // Hide tools initially
                document.getElementById('color-definition-area').style.display = 'none';
                document.getElementById('add-variant-btn').style.display = 'none';

                addVariant(true);
                fillVariantData(0, dbVariants[0]);
            } else {
                // COLOR MODE
                document.getElementById('colorYes').checked = true;
                document.getElementById('color-definition-area').style.display = 'block';
                document.getElementById('add-variant-btn').style.display = 'flex';

                // Rebuild Colors
                dbVariants.forEach(v => {
                    if (v.color === 'Standard') return;
                    let hex = '#333';
                    const option = document.querySelector(`#color-picker option[value="${v.color}"]`);
                    if (option) hex = option.getAttribute('data-hex');

                    if (!availableColors.some(c => c.name === v.color)) {
                        availableColors.push({
                            name: v.color,
                            hex: hex
                        });
                    }
                });
                renderMasterColors();
                updateDefaultColorDropdown();
                if (document.getElementById('default-color-selector')) {
                    document.getElementById('default-color-selector').value = dbDefaultColor;
                }

                // Cards
                usedColors = [];
                dbVariants.forEach(v => {
                    let colorToPass = (v.color === 'Standard') ? null : v.color;
                    addVariant(false, colorToPass);
                    fillVariantData(variantCount - 1, v);
                });
            }
        } else {
            toggleColorMode(false);
        }
    });

    // Helper: Fill data
    function fillVariantData(idx, data) {
        // Fill Sizes
        const sizeCont = document.getElementById(`sizes-${idx}`);
        sizeCont.innerHTML = '';
        if (data.sizes.length > 0) {
            data.sizes.forEach(s => addSizeRow(idx, s.size, s.stock_quantity));
        } else {
            addSizeRow(idx);
        }

        // Fill Images
        const imgCont = document.getElementById(`imgs-${idx}`);
        const triggerLabel = imgCont.querySelector('.upload-trigger-box');

        if (data.images.length > 0) {
            data.images.forEach(img => {
                const div = document.createElement('div');
                div.className = 'img-preview-item';
                div.innerHTML = `
                    <img src="../${img.image_path}">
                    <span class="img-remove-x" onclick="markImageDelete(this, ${img.id})">×</span>
                    <input type="radio" class="primary-radio-btn" name="variants[${idx}][primary_img]" value="${img.id}" ${img.is_primary == 1 ? 'checked' : ''} title="Set Primary">
                `;
                imgCont.insertBefore(div, triggerLabel);
            });
        }
    }

    // --- 3. LOGIC FOR SWITCHING MODES (FIXED) ---

    function toggleColorMode(toColorMode) {
        const area = document.getElementById('color-definition-area');
        const btn = document.getElementById('add-variant-btn');
        const wrapper = document.getElementById('variants-wrapper');

        if (toColorMode) {
            // === SWITCHING TO: COLORS ===
            area.style.display = 'block';
            btn.style.display = 'flex';

            const firstCard = wrapper.firstElementChild;
            if (firstCard) {
                // We have a card (Standard), convert it to Color Dropdown
                const idx = firstCard.id.split('-')[1];
                const headerDiv = firstCard.querySelector('.variant-header');

                // Swap Header
                headerDiv.innerHTML = getVariantHeaderHTML(idx, false);

                // Reset used colors so the new dropdown works
                usedColors = [];
                updateAllDropdowns();
            } else {
                addVariant(false);
            }

        } else {
            // === SWITCHING TO: NO COLORS (STANDARD) ===
            const cards = wrapper.querySelectorAll('.variant-group-card');

            if (cards.length > 1) {
                if (!confirm("Merging to 'No Colors' will combine all images into one group. Extra size groups will be removed. Continue?")) {
                    document.getElementById('colorYes').checked = true;
                    return;
                }
            }

            area.style.display = 'none';
            btn.style.display = 'none';

            let targetCardId = null;
            let targetImgContainer = null;
            let targetIdx = null;

            // Loop to Merge
            cards.forEach((card, i) => {
                const currentIdx = card.id.split('-')[1];

                if (i === 0) {
                    // This is the TARGET (Survivor)
                    targetCardId = card.id;
                    targetIdx = currentIdx;
                    targetImgContainer = card.querySelector('.img-preview-container');

                    // Update Header to Standard
                    const headerDiv = card.querySelector('.variant-header');
                    headerDiv.innerHTML = getVariantHeaderHTML(currentIdx, true);

                } else {
                    // This is a SOURCE (Will be deleted)
                    const sourceImgContainer = card.querySelector('.img-preview-container');
                    const images = sourceImgContainer.querySelectorAll('.img-preview-item');
                    const targetLabel = targetImgContainer.querySelector('.upload-trigger-box');

                    images.forEach(imgDiv => {
                        // 1. Move the Div
                        targetImgContainer.insertBefore(imgDiv, targetLabel);

                        // 2. RENAME INPUTS (Crucial Fix)
                        // Change variants[1][images] -> variants[0][images]
                        // Change variants[1][primary_img] -> variants[0][primary_img]
                        const inputs = imgDiv.querySelectorAll('input');
                        inputs.forEach(inp => {
                            if (inp.name) {
                                inp.name = inp.name.replace(/variants\[\d+\]/, `variants[${targetIdx}]`);
                            }
                        });
                    });

                    // Remove the old card
                    card.remove();
                }
            });

            if (!targetCardId) {
                addVariant(true);
            } else {
                // FIX: Ensure only ONE primary radio is checked after merge
                enforceSinglePrimary(targetIdx);
                // FIX: Re-index "new_" radio values to prevent collisions
                reindexRadios(targetIdx);
            }

            usedColors = [];
        }
    }

    // Helper: Enforce Single Primary Radio
    function enforceSinglePrimary(idx) {
        const container = document.getElementById(`imgs-${idx}`);
        // Get all checked radios
        const checked = container.querySelectorAll('input[type="radio"]:checked');
        if (checked.length > 1) {
            // Keep the first one, uncheck others
            for (let i = 1; i < checked.length; i++) {
                checked[i].checked = false;
            }
        }
    }

    function getVariantHeaderHTML(idx, isStandard) {
        if (isStandard) {
            return `
                <h6 class="mb-0 text-muted"><i class="fas fa-box mr-2"></i> Standard Version (No Color)</h6>
                <input type="hidden" name="variants[${idx}][color]" value="Standard">
            `;
        } else {
            return `
                <div class="d-flex justify-content-between align-items-center w-100">
                    <div style="width: 250px;">
                        <label class="input-label mb-1">Select Color Variant</label>
                        <select name="variants[${idx}][color]" class="clean-input variant-color-select" data-idx="${idx}" onchange="handleColorSelect(this)" required>
                            <option value="">Select Color</option>
                        </select>
                    </div>
                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeVariant(${idx})"><i class="fas fa-trash"></i> Remove Group</button>
                </div>
            `;
        }
    }

    function addVariant(isStandard = false, preSelectColor = null) {
        if (!isStandard && availableColors.length === 0) {
            alert("Please define at least one color first in Step 1.");
            return;
        }

        const idx = variantCount;
        const headerHTML = getVariantHeaderHTML(idx, isStandard);

        const html = `
        <div class="variant-group-card" id="var-${idx}">
            <div class="variant-header">${headerHTML}</div>
            <div class="row">
                <div class="col-lg-5 col-md-12 border-right-lg mobile-mb-4">
                    <label class="input-label bg-light p-2 rounded mb-2">Sizes & Stock</label>
                    <div id="sizes-${idx}"></div>
                    <button type="button" class="btn-soft mt-2" onclick="addSizeRow(${idx})"><i class="fas fa-plus"></i> Add Size</button>
                </div>
                <div class="col-lg-7 col-md-12">
                    <label class="input-label bg-light p-2 rounded mb-2">Images </label>
                    <div class="img-preview-container" id="imgs-${idx}">
                        <label class="upload-trigger-box" for="upl-${idx}">
                            <div class="text-center"><i class="fas fa-plus fa-lg mb-1"></i><br><small>Add <br>(670 x 800)</small></div>
                            <input type="file" id="upl-${idx}" name="variants[${idx}][images][]" multiple accept="image/*" style="display:none;" onchange="handleImg(this, ${idx})">
                        </label>
                    </div>
                </div>
            </div>
        </div>`;

        document.getElementById('variants-wrapper').insertAdjacentHTML('beforeend', html);

        if (!isStandard) {
            updateAllDropdowns();
            if (preSelectColor) {
                const sel = document.querySelector(`#var-${idx} select.variant-color-select`);
                if (sel) {
                    sel.value = preSelectColor;
                    sel.setAttribute('data-prev', preSelectColor);
                    usedColors.push(preSelectColor);
                    updateAllDropdowns();
                }
            }
        }

        addSizeRow(idx);
        variantCount++;
    }

    // --- IMAGES ---

    function handleImg(input, idx) {
        const container = document.getElementById(`imgs-${idx}`);
        const triggerLabel = container.querySelector('.upload-trigger-box');

        if (input.files && input.files.length > 0) {
            Array.from(input.files).forEach((file, i) => {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const div = document.createElement('div');
                    div.className = 'img-preview-item';

                    const dt = new DataTransfer();
                    dt.items.add(file);
                    const singleFileInput = document.createElement('input');
                    singleFileInput.type = 'file';
                    singleFileInput.name = `variants[${idx}][images][]`;
                    singleFileInput.files = dt.files;
                    singleFileInput.style.display = 'none';

                    div.innerHTML = `
                        <img src="${e.target.result}">
                        <span class="img-remove-x" onclick="removeNewImage(this, ${idx})">×</span>
                        <input type="radio" class="primary-radio-btn" name="variants[${idx}][primary_img]" value="new_TEMP">
                    `;

                    div.appendChild(singleFileInput);
                    container.insertBefore(div, triggerLabel);
                    reindexRadios(idx);
                }
                reader.readAsDataURL(file);
            });

            input.value = '';
        }
    }

    function removeNewImage(btn, idx) {
        btn.parentElement.remove();
        reindexRadios(idx);
    }

    function markImageDelete(btn, id) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'delete_image_ids[]';
        input.value = id;
        document.getElementById('productForm').appendChild(input);
        btn.parentElement.remove();
    }

    function reindexRadios(idx) {
        const container = document.getElementById(`imgs-${idx}`);
        const radios = container.querySelectorAll('input.primary-radio-btn[value^="new_"]');
        radios.forEach((radio, i) => {
            radio.value = "new_" + i;
        });
    }

    // --- STANDARD UTILS ---

    function addColorToMaster() {
        const select = document.getElementById('color-picker');
        const name = select.value;
        const hex = select.options[select.selectedIndex].getAttribute('data-hex');

        if (name && !availableColors.some(c => c.name === name)) {
            availableColors.push({
                name,
                hex
            });
            renderMasterColors();
            updateDefaultColorDropdown();
            updateAllDropdowns();
        }
    }

    function renderMasterColors() {
        document.getElementById('master-color-list').innerHTML = availableColors.map(c => `
            <div class="color-tag">
                <span class="color-swatch" style="background:${c.hex}"></span>
                ${c.name}
                <i class="fas fa-times remove-color-btn" onclick="removeMasterColor('${c.name}')"></i>
            </div>
        `).join('');
    }

    function removeMasterColor(name) {
        if (usedColors.includes(name)) {
            alert("Cannot remove a color that is in use.");
            return;
        }
        availableColors = availableColors.filter(c => c.name !== name);
        renderMasterColors();
        updateDefaultColorDropdown();
        updateAllDropdowns();
    }

    function handleColorSelect(select) {
        const prevVal = select.getAttribute('data-prev');
        const newVal = select.value;
        if (prevVal) usedColors = usedColors.filter(c => c !== prevVal);
        if (newVal) usedColors.push(newVal);
        select.setAttribute('data-prev', newVal);
        updateAllDropdowns();
    }

    function removeVariant(idx) {
        const select = document.querySelector(`#var-${idx} select.variant-color-select`);
        if (select && select.value) {
            usedColors = usedColors.filter(c => c !== select.value);
        }
        document.getElementById(`var-${idx}`).remove();
        updateAllDropdowns();
    }

    function updateAllDropdowns() {
        const allSelects = document.querySelectorAll('.variant-color-select');
        allSelects.forEach(select => {
            const currentValue = select.value;
            let html = '<option value="">Select Color</option>';
            availableColors.forEach(c => {
                if (!usedColors.includes(c.name) || c.name === currentValue) {
                    html += `<option value="${c.name}" ${c.name === currentValue ? 'selected' : ''}>${c.name}</option>`;
                }
            });
            select.innerHTML = html;
        });
    }

    function addSizeRow(idx, name = '', qty = '') {
        const html = `<div class="d-flex gap-2 mb-2 align-items-center"><input type="text" name="variants[${idx}][sizes][name][]" value="${name}" class="clean-input" placeholder="Size" style="width:100px; height:48px;" required autocomplete="off"><input type="number" name="variants[${idx}][sizes][qty][]" value="${qty}" class="clean-input" placeholder="Qty" style="width:100px; height:48px;" required autocomplete="off"><button type="button" class="btn btn-sm text-danger" onclick="this.parentElement.remove()" style="height:48px; width:40px; display:flex; justify-content:center; align-items:center;"><i class="fas fa-times"></i></button></div>`;
        document.getElementById(`sizes-${idx}`).insertAdjacentHTML('beforeend', html);
    }

    function updateDefaultColorDropdown() {
        if (!document.getElementById('default-color-selector')) return;
        const select = document.getElementById('default-color-selector');
        const currentValue = select.value;
        let html = '<option value="">-- Select Default --</option>';
        availableColors.forEach(c => {
            let selected = (c.name === currentValue) ? 'selected' : '';
            html += `<option value="${c.name}" ${selected}>${c.name}</option>`;
        });
        select.innerHTML = html;
    }

    function addSpecRow(key = '', val = '') {
        const html = `<div class="d-flex gap-2 mb-2"><input type="text" name="spec_key[]" value="${key}" class="clean-input" placeholder="Label (e.g. Fabric)" autocomplete="off"><input type="text" name="spec_val[]" value="${val}" class="clean-input" placeholder="Value (e.g. Cottton)" autocomplete="off"><button type="button" class="btn btn-sm text-danger" onclick="this.parentElement.remove()" style="height:48px; width:40px; display:flex; justify-content:center; align-items:center;"><i class="fas fa-times"></i></button></div>`;
        document.getElementById('specs-container').insertAdjacentHTML('beforeend', html);
    }
</script>

<?php include 'includes/admin_footer.php'; ?>