<?php
require_once 'includes/db_config.php';
check_login();

// --- 1. HANDLE FORM SUBMISSION ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_product'])) {

    // A. General Info
    $name = sanitize_input($_POST['name']);
    $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $name)));
    $short = sanitize_input($_POST['short_description']);
    $desc = $_POST['description']; // Full HTML/Text
    $cat_id = (int)$_POST['category_id'];
    $price = (float)$_POST['price'];
    $sale = !empty($_POST['sale_price']) ? (float)$_POST['sale_price'] : NULL;
    $sku = sanitize_input($_POST['sku']);
    $active = isset($_POST['is_active']) ? 1 : 0;

    // B. Specifications (Key-Value)
    $specs = [];
    if (isset($_POST['spec_key'])) {
        for ($i = 0; $i < count($_POST['spec_key']); $i++) {
            if (!empty($_POST['spec_key'][$i])) {
                $specs[$_POST['spec_key'][$i]] = $_POST['spec_val'][$i];
            }
        }
    }
    $info = json_encode($specs);

    // C. Insert Product Base (without default color initially)
    $stmt = $conn->prepare("INSERT INTO products (name, slug, sku, category_id, short_description, description, additional_info, price, sale_price, is_active) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("sssisssddi", $name, $slug, $sku, $cat_id, $short, $desc, $info, $price, $sale, $active);
    $stmt->execute();
    $pid = $stmt->insert_id;

    // Capture the default color
    $default_color = sanitize_input($_POST['default_color_group'] ?? 'Standard');

    // D. Handle Variations (Color Groups)
    if (isset($_POST['variants'])) {
        foreach ($_POST['variants'] as $idx => $var) {

            // Determine Color Name
            // If "No" colors selected, we use 'Standard'. Else use the dropdown value.
            $hasColor = $_POST['has_color_option'] ?? '0';
            $colorName = ($hasColor == '1') ? sanitize_input($var['color']) : 'Standard';

            // 1. Insert Sizes
            if (isset($var['sizes']['name'])) {
                foreach ($var['sizes']['name'] as $si => $sName) {
                    $sQty = (int)$var['sizes']['qty'][$si];
                    if (!empty($sName)) {
                        $stmt = $conn->prepare("INSERT INTO product_sizes (product_id, color_group, size, stock_quantity) VALUES (?, ?, ?, ?)");
                        $stmt->bind_param("issi", $pid, $colorName, $sName, $sQty);
                        $stmt->execute();
                    }
                }
            }

            // 2. Insert Images with proper primary image handling
            if (isset($_FILES['variants']['name'][$idx]['images'])) {
                $fileAry = $_FILES['variants'];
                $count = count($fileAry['name'][$idx]['images']);

                // Get selected primary image (could be "new_X" for new images or numeric ID for existing)
                $selected_primary = $var['primary_img'] ?? '';

                // Handle primary image logic
                $primary_set_for_color = false;

                for ($k = 0; $k < $count; $k++) {
                    if ($fileAry['error'][$idx]['images'][$k] === 0) {
                        $tmpName = $fileAry['tmp_name'][$idx]['images'][$k];
                        $origName = $fileAry['name'][$idx]['images'][$k];
                        $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));

                        $newName = $slug . '_' . $colorName . '_' . uniqid() . '.' . $ext;
                        $target = '../assets/uploads/products/' . $newName;

                        if (move_uploaded_file($tmpName, $target)) {
                            $dbPath = 'assets/uploads/products/' . $newName;

                            // Determine if this image should be primary
                            $isPrimary = 0;

                            if ($selected_primary === "new_$k") {
                                // This new image was selected as primary
                                $isPrimary = 1;
                                $primary_set_for_color = true;
                            } elseif ($selected_primary === '' && $k === 0 && !$primary_set_for_color) {
                                // No primary selected, first image becomes primary
                                $isPrimary = 1;
                                $primary_set_for_color = true;
                            }

                            $stmt = $conn->prepare("INSERT INTO product_images (product_id, image_path, color_group, is_primary) VALUES (?, ?, ?, ?)");
                            $stmt->bind_param("issi", $pid, $dbPath, $colorName, $isPrimary);
                            $stmt->execute();
                        }
                    }
                }

                // If no primary was set for this color (shouldn't happen), set the first one
                if (!$primary_set_for_color) {
                    // Get the last inserted ID for this color and set it as primary
                    $update_sql = "UPDATE product_images SET is_primary = 1 
                      WHERE product_id = $pid 
                      AND color_group = '$colorName' 
                      ORDER BY id ASC LIMIT 1";
                    $conn->query($update_sql);
                }
            }
        }
    }

    // E. Final Step: Update the product with the default color
    $update_default = $conn->prepare("UPDATE products SET default_color_group = ? WHERE id = ?");
    $update_default->bind_param("si", $default_color, $pid);
    $update_default->execute();

    $_SESSION['flash_message'] = '<div class="alert alert-success">Product added successfully!</div>';
    header("Location: products.php");
    exit;
}

// Fetch categories (excluding virtual ones like New Arrivals/Best Sellers/On Sale)
$cats = $conn->query("SELECT id, name FROM categories WHERE slug NOT IN ('new-arrivals', 'best-sellers', 'on-sale') ORDER BY name ASC");
$page_title = "Add Product";
include 'includes/admin_header.php';
?>

<div class="container-fluid">

    <!-- 1. BACK NAVIGATION -->
    <div class="page-header-nav">
        <a href="products.php"><i class="fas fa-arrow-left"></i></a>
        <h2>Add New Product</h2>
    </div>

    <form id="productForm" method="POST" enctype="multipart/form-data">

        <div class="row">
            <!-- 2. GENERAL INFO (Left) -->
            <div class="col-lg-12">
                <div class="product-card h-100">
                    <h5 class="section-title">1. General Information</h5>

                    <!-- Row 1: Product Name + Category -->
                    <div class="row">

                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Product Name <span class="text-danger">*</span></label>
                                <input type="text" name="name" class="clean-input" placeholder="e.g. Floral Maxi Dress" required>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Category <span class="text-danger">*</span></label>
                                <select name="category_id" class="clean-input" required>
                                    <option value="">-- Select Category --</option>
                                    <?php while ($c = $cats->fetch_assoc()): ?>
                                        <option value="<?= $c['id'] ?>"><?= $c['name'] ?></option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                        </div>

                    </div>

                    <!-- Row 2: Short Description + Detailed Description -->
                    <div class="row">

                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Short Description</label>
                                <textarea name="short_description" class="clean-input" rows="4"
                                    placeholder="Brief summary displayed on listings..."></textarea>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Detailed Description</label>
                                <textarea name="description" class="clean-input" rows="4"
                                    placeholder="Full details about the product..."></textarea>
                            </div>
                        </div>

                    </div>

                    <!-- Additional Specifications (Full Width) -->
                    <div class="mt-2 pt-3 border-top">

                        <label class="input-label mb-2">Additional Specifications</label>

                        <div id="specs-container"></div>

                        <button type="button" class="btn-soft mt-2" onclick="addSpecRow()">
                            <i class="fas fa-plus"></i> Add Specification
                        </button>

                    </div>

                </div>
            </div>

            <!-- 3. PRICING & ORG (Right) -->
            <div class="col-lg-12 mt-3">
                <div class="product-card h-100">
                    <h5 class="section-title">2. Pricing & Organization</h5>

                    <!-- Row 1: Base Price + Sale Price -->
                    <div class="row">

                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Base Price (₹) <span class="text-danger">*</span></label>
                                <input type="number" name="price" class="clean-input" step="0.01" placeholder="0.00" required>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">Sale Price (₹)</label>
                                <input type="number" name="sale_price" class="clean-input" step="0.01" placeholder="0.00">
                            </div>
                        </div>

                    </div>

                    <hr>

                    <!-- Row 2: SKU + Active Status -->
                    <div class="row align-items-center">

                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="input-label">SKU (Stock Unit ID)</label>
                                <input type="text" name="sku" class="clean-input" placeholder="e.g. DRESS-001">
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="custom-control custom-switch mt-4">
                                <input type="checkbox" class="custom-control-input" id="isActive" name="is_active" checked>
                                <label class="custom-control-label font-weight-bold" for="isActive">
                                    Active Status
                                </label>
                            </div>
                        </div>

                    </div>

                </div>
            </div>

        </div>

        <!-- 4. PRODUCT VARIATIONS (Full Width) -->
        <div class="row mt-3">
            <div class="col-12">
                <div class="product-card">
                    <h5 class="section-title">3. Product Variations</h5>

                    <!-- Color Question -->
                    <div class="bg-light p-3 rounded mb-4 border d-flex align-items-center gap-3">
                        <span class="font-weight-bold mr-3">Does this product have different colors?</span>
                        <div class="custom-control custom-radio custom-control-inline">
                            <input type="radio" id="colorYes" name="has_color_option" value="1" class="custom-control-input" onchange="toggleColorMode(true)">
                            <label class="custom-control-label" for="colorYes">Yes</label>
                        </div>
                        <div class="custom-control custom-radio custom-control-inline">
                            <input type="radio" id="colorNo" name="has_color_option" value="0" class="custom-control-input" checked onchange="toggleColorMode(false)">
                            <label class="custom-control-label" for="colorNo">No</label>
                        </div>
                    </div>

                    <!-- Color Definition Area (Hidden by default) -->
                    <div id="color-definition-area" style="display:none;" class="mb-4">

                        <div class="row">

                            <!-- STEP 1: ADD COLORS -->
                            <div class="col-md-6">
                                <div class="pl-3 border-left">

                                    <label class="input-label">Step 1: Add Available Colors</label>

                                    <div class="d-flex gap-2 align-items-center mb-2">
                                        <select id="color-picker" class="clean-input" style="width: 200px;">
                                            <option value="">Select Color</option>

                                            <!-- Basics -->
                                            <option value="Red" data-hex="#da0103">Red</option>
                                            <option value="Blue" data-hex="#0000FF">Blue</option>
                                            <option value="Green" data-hex="#008000">Green</option>
                                            <option value="Black" data-hex="#000000">Black</option>
                                            <option value="White" data-hex="#FFFFFF">White</option>
                                            <option value="Grey" data-hex="#808080">Grey</option>

                                            <!-- Boutique / Premium Colors -->
                                            <option value="Maroon" data-hex="#800000">Maroon</option>
                                            <option value="Wine" data-hex="#722f37">Wine</option>
                                            <option value="Navy" data-hex="#000080">Navy</option>
                                            <option value="Teal" data-hex="#008080">Teal</option>
                                            <option value="Mustard" data-hex="#e1ad01">Mustard</option>
                                            <option value="Purple" data-hex="#800080">Purple</option>
                                            <option value="Brown" data-hex="#8b4513">Brown</option>

                                            <!-- Metallic & Luxury -->
                                            <option value="Gold" data-hex="#d4af37">Gold</option>
                                            <option value="Silver" data-hex="#c0c0c0">Silver</option>
                                            <option value="Ivory" data-hex="#fffff0">Ivory</option>
                                            <option value="Cream" data-hex="#fffdd0">Cream</option>
                                            <option value="Beige" data-hex="#f5f5dc">Beige</option>

                                            <!-- Pastels & Brights -->
                                            <option value="Peach" data-hex="#ffdab9">Peach</option>
                                            <option value="Pink" data-hex="#ffc0cb">Pink</option>
                                            <option value="Magenta" data-hex="#ff00ff">Magenta</option>
                                            <option value="Yellow" data-hex="#ffff00">Yellow</option>
                                            <option value="Orange" data-hex="#ffa500">Orange</option>
                                            <option value="Lavender" data-hex="#e6e6fa">Lavender</option>
                                            <option value="Turquoise" data-hex="#40e0d0">Turquoise</option>
                                        </select>

                                        <button type="button" class="btn btn-dark" onclick="addColorToMaster()">
                                            Add
                                        </button>
                                    </div>

                                    <div id="master-color-list" class="d-flex flex-wrap gap-2 mb-3"></div>

                                </div>
                            </div>

                            <!-- STEP 2: DEFAULT COLOR -->
                            <div class="col-md-6">
                                <div class="form-group pt-3 pt-md-0">

                                    <label class="input-label">Step 2: Set Default Display Color</label>

                                    <select name="default_color_group" id="default-color-selector" class="clean-input" style="width: 250px;">
                                        <!-- Options populated by JS -->
                                    </select>

                                    <!-- <small class="text-muted">
                                        This color is shown first on the product page.
                                    </small> -->

                                </div>
                            </div>

                        </div>

                    </div>


                    <!-- Variants Container -->
                    <div id="variants-wrapper"></div>

                    <!-- Add Variant Button (Hidden if No Color) -->
                    <button type="button" class="btn-soft w-100 justify-content-center mt-3" id="add-variant-btn" onclick="addVariant()" style="display:none; border-style:dashed;">
                        <i class="fas fa-plus"></i> Add Another Color Variation
                    </button>
                </div>
            </div>
        </div>

        <!-- 5. SAVE BUTTON BAR -->
        <div class="form-footer-action">
            <button type="submit" name="add_product" class="btn-save-product">Save Product</button>
        </div>

    </form>
</div>

<!-- JAVASCRIPT LOGIC -->
<script>
    // --- 1. INITIALIZATION DATA FROM PHP ---
    // (Use empty arrays/defaults if variables aren't defined to prevent JS errors on Add Page)
    const dbVariants = <?= isset($js_variants) ? json_encode($js_variants) : '[]' ?>;
    const dbSpecs = <?= isset($specs) ? json_encode($specs) : '[]' ?>;
    const isStandardMode = <?= (isset($is_standard_mode) && $is_standard_mode) ? 'true' : 'false' ?>;
    const dbDefaultColor = "<?= $current_default_color ?? '' ?>";

    let variantCount = 0;
    let availableColors = [];
    let usedColors = [];

    // --- 2. STARTUP SEQUENCE ---
    document.addEventListener('DOMContentLoaded', function() {

        // A. Specs
        if (Object.keys(dbSpecs).length > 0) {
            for (const [key, value] of Object.entries(dbSpecs)) {
                addSpecRow(key, value);
            }
        } else {
            addSpecRow();
        }

        // B. Variants Init
        // If we are on the ADD page, dbVariants might be empty or undefined
        if (typeof dbVariants !== 'undefined' && dbVariants.length > 0) {

            if (isStandardMode) {
                // Standard Mode (Edit Page)
                document.getElementById('colorNo').checked = true;
                toggleColorMode(false);
                fillVariantData(0, dbVariants[0]);
            } else {
                // Color Mode (Edit Page)
                document.getElementById('colorYes').checked = true;
                document.getElementById('color-definition-area').style.display = 'block';
                document.getElementById('add-variant-btn').style.display = 'flex';

                // Rebuild Available Colors
                dbVariants.forEach(v => {
                    if (v.color === 'Standard') return;
                    let hex = '#333';
                    const option = document.querySelector(`#color-picker option[value="${v.color}"]`);
                    if (option) hex = option.getAttribute('data-hex');

                    if (!availableColors.some(c => c.name === v.color)) {
                        availableColors.push({
                            name: v.color,
                            hex: hex
                        });
                    }
                });
                renderMasterColors();
                updateDefaultColorDropdown();
                if (document.getElementById('default-color-selector')) {
                    document.getElementById('default-color-selector').value = dbDefaultColor;
                }

                // Create Variant Cards
                usedColors = [];
                dbVariants.forEach(v => {
                    let colorToPass = (v.color === 'Standard') ? null : v.color;
                    addVariant(false, colorToPass);
                    fillVariantData(variantCount - 1, v);
                });
            }
        } else {
            // Default State for ADD Page
            toggleColorMode(false);
        }
    });

    // Helper: Fill data (For Edit Page)
    function fillVariantData(idx, data) {
        const sizeCont = document.getElementById(`sizes-${idx}`);
        sizeCont.innerHTML = '';
        if (data.sizes.length > 0) {
            data.sizes.forEach(s => addSizeRow(idx, s.size, s.stock_quantity));
        } else {
            addSizeRow(idx);
        }

        const imgCont = document.getElementById(`imgs-${idx}`);
        const triggerLabel = imgCont.querySelector('.upload-trigger-box');

        if (data.images.length > 0) {
            data.images.forEach(img => {
                const div = document.createElement('div');
                div.className = 'img-preview-item';
                div.innerHTML = `
                    <img src="../${img.image_path}">
                    <span class="img-remove-x" onclick="markImageDelete(this, ${img.id})">×</span>
                    <input type="radio" class="primary-radio-btn" name="variants[${idx}][primary_img]" value="${img.id}" ${img.is_primary == 1 ? 'checked' : ''} title="Set Primary">
                `;
                imgCont.insertBefore(div, triggerLabel);
            });
        }
    }

    // --- 3. LOGIC FOR SWITCHING MODES ---
    function toggleColorMode(toColorMode) {
        const area = document.getElementById('color-definition-area');
        const btn = document.getElementById('add-variant-btn');
        const wrapper = document.getElementById('variants-wrapper');

        if (toColorMode) {
            // TO COLORS
            area.style.display = 'block';
            btn.style.display = 'flex';

            const firstCard = wrapper.firstElementChild;
            if (firstCard) {
                const idx = firstCard.id.split('-')[1];
                const headerDiv = firstCard.querySelector('.variant-header');
                headerDiv.innerHTML = getVariantHeaderHTML(idx, false);
                usedColors = [];
                updateAllDropdowns();
            } else {
                addVariant(false);
            }
        } else {
            // TO STANDARD
            const cards = wrapper.querySelectorAll('.variant-group-card');
            if (cards.length > 1) {
                if (!confirm("Merging to 'No Colors' will combine all images. Continue?")) {
                    document.getElementById('colorYes').checked = true;
                    return;
                }
            }

            area.style.display = 'none';
            btn.style.display = 'none';

            let targetCardId = null;
            let targetImgContainer = null;

            cards.forEach((card, i) => {
                if (i === 0) {
                    targetCardId = card.id;
                    targetImgContainer = card.querySelector('.img-preview-container');
                    const headerDiv = card.querySelector('.variant-header');
                    const idx = card.id.split('-')[1];
                    headerDiv.innerHTML = getVariantHeaderHTML(idx, true);
                } else {
                    // Merge logic
                    const images = card.querySelectorAll('.img-preview-item');
                    const targetLabel = targetImgContainer.querySelector('.upload-trigger-box');
                    images.forEach(imgDiv => {
                        targetImgContainer.insertBefore(imgDiv, targetLabel);
                    });
                    card.remove();
                }
            });

            if (!targetCardId) addVariant(true);
            usedColors = [];
            // After merge, re-index radios in the surviving card to prevent Primary mismatch
            if (targetCardId) {
                const idx = targetCardId.split('-')[1];
                reindexRadios(idx);
            }
        }
    }

    function getVariantHeaderHTML(idx, isStandard) {
        if (isStandard) {
            return `
                <h6 class="mb-0 text-muted"><i class="fas fa-box mr-2"></i> Standard Version (No Color)</h6>
                <input type="hidden" name="variants[${idx}][color]" value="Standard">
            `;
        } else {
            return `
                <div class="d-flex justify-content-between align-items-center w-100">
                    <div style="width: 250px;">
                        <label class="input-label mb-1">Select Color Variant</label>
                        <select name="variants[${idx}][color]" class="clean-input variant-color-select" data-idx="${idx}" onchange="handleColorSelect(this)" required>
                            <option value="">Select Color</option>
                        </select>
                    </div>
                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeVariant(${idx})"><i class="fas fa-trash"></i> Remove Group</button>
                </div>
            `;
        }
    }

    function addVariant(isStandard = false, preSelectColor = null) {
        if (!isStandard && availableColors.length === 0) {
            alert("Please define at least one color first in Step 1.");
            return;
        }

        const idx = variantCount;
        const headerHTML = getVariantHeaderHTML(idx, isStandard);

        const html = `
        <div class="variant-group-card" id="var-${idx}">
            <div class="variant-header">${headerHTML}</div>
            <div class="row">
                <div class="col-lg-5 col-md-12 border-right-lg mobile-mb-4">
                    <label class="input-label bg-light p-2 rounded mb-2">Sizes & Stock</label>
                    <div id="sizes-${idx}"></div>
                    <button type="button" class="btn-soft mt-2" onclick="addSizeRow(${idx})"><i class="fas fa-plus"></i> Add Size</button>
                </div>
                <div class="col-lg-7 col-md-12">
                    <label class="input-label bg-light p-2 rounded mb-2">Images</label>
                    <div class="img-preview-container" id="imgs-${idx}">
                        <label class="upload-trigger-box" for="upl-${idx}">
                            <div class="text-center"><i class="fas fa-plus fa-lg mb-1"></i><br><small>Add <br>(670 x 800)</small></div>
                            <!-- Note: 'multiple' is removed to force clean handling via JS, 
                                 but user can select multiple because we handle it in handleImg -->
                            <input type="file" id="upl-${idx}" name="variants[${idx}][images][]" multiple accept="image/*" style="display:none;" onchange="handleImg(this, ${idx})">
                        </label>
                    </div>
                </div>
            </div>
        </div>`;

        document.getElementById('variants-wrapper').insertAdjacentHTML('beforeend', html);

        if (!isStandard) {
            updateAllDropdowns();
            if (preSelectColor) {
                const sel = document.querySelector(`#var-${idx} select.variant-color-select`);
                if (sel) {
                    sel.value = preSelectColor;
                    sel.setAttribute('data-prev', preSelectColor);
                    usedColors.push(preSelectColor);
                    updateAllDropdowns();
                }
            }
        }

        addSizeRow(idx);
        variantCount++;
    }

    // --- CRITICAL FIX: IMAGE HANDLING ---

    // 1. Handle New Uploads (Splits multiple files into individual inputs)
    function handleImg(input, idx) {
        const container = document.getElementById(`imgs-${idx}`);
        const triggerLabel = container.querySelector('.upload-trigger-box');

        if (input.files && input.files.length > 0) {
            Array.from(input.files).forEach((file, i) => {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const div = document.createElement('div');
                    div.className = 'img-preview-item';

                    // Create a dedicated input for THIS single file
                    const dt = new DataTransfer();
                    dt.items.add(file);
                    const singleFileInput = document.createElement('input');
                    singleFileInput.type = 'file';
                    singleFileInput.name = `variants[${idx}][images][]`;
                    singleFileInput.files = dt.files;
                    singleFileInput.style.display = 'none';

                    // HTML for Preview
                    div.innerHTML = `
                        <img src="${e.target.result}">
                        <span class="img-remove-x" onclick="removeNewImage(this, ${idx})">×</span>
                        <input type="radio" class="primary-radio-btn" name="variants[${idx}][primary_img]" value="new_TEMP">
                    `;

                    // Append the actual file input to the div
                    div.appendChild(singleFileInput);

                    container.insertBefore(div, triggerLabel);

                    // Re-index all new uploads immediately so Primary matches PHP array
                    reindexRadios(idx);
                }
                reader.readAsDataURL(file);
            });

            // Reset the trigger input so the same files can be selected again if needed
            input.value = '';
        }
    }

    // 2. Remove New Image & Re-index
    function removeNewImage(btn, idx) {
        // Since the input is inside the div, removing the div removes the file from submission
        btn.parentElement.remove();
        // Vital: Re-index remaining radios so PHP array indices match
        reindexRadios(idx);
    }

    // 3. Mark Existing DB Image for deletion
    function markImageDelete(btn, id) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'delete_image_ids[]';
        input.value = id;
        // Append to form, not the image container (to keep styling clean)
        document.getElementById('productForm').appendChild(input);
        btn.parentElement.remove();
        // No need to re-index radios for existing images as they use IDs
    }

    // 4. Re-index Helper
    function reindexRadios(idx) {
        const container = document.getElementById(`imgs-${idx}`);
        // Only select radios that are for NEW uploads (value starts with new_)
        const radios = container.querySelectorAll('input.primary-radio-btn[value^="new_"]');

        radios.forEach((radio, i) => {
            // This ensures the first visual "new" image corresponds to Index 0 in PHP's received array
            radio.value = "new_" + i;
        });
    }

    // --- STANDARD UTILS ---

    function addColorToMaster() {
        const select = document.getElementById('color-picker');
        const name = select.value;
        const hex = select.options[select.selectedIndex].getAttribute('data-hex');

        if (name && !availableColors.some(c => c.name === name)) {
            availableColors.push({
                name,
                hex
            });
            renderMasterColors();
            updateDefaultColorDropdown();
            updateAllDropdowns();
        }
    }

    function renderMasterColors() {
        document.getElementById('master-color-list').innerHTML = availableColors.map(c => `
            <div class="color-tag">
                <span class="color-swatch" style="background:${c.hex}"></span>
                ${c.name}
                <i class="fas fa-times remove-color-btn" onclick="removeMasterColor('${c.name}')"></i>
            </div>
        `).join('');
    }

    function removeMasterColor(name) {
        if (usedColors.includes(name)) {
            alert("Cannot remove a color that is in use.");
            return;
        }
        availableColors = availableColors.filter(c => c.name !== name);
        renderMasterColors();
        updateDefaultColorDropdown();
        updateAllDropdowns();
    }

    function handleColorSelect(select) {
        const prevVal = select.getAttribute('data-prev');
        const newVal = select.value;
        if (prevVal) usedColors = usedColors.filter(c => c !== prevVal);
        if (newVal) usedColors.push(newVal);
        select.setAttribute('data-prev', newVal);
        updateAllDropdowns();
    }

    function removeVariant(idx) {
        const select = document.querySelector(`#var-${idx} select.variant-color-select`);
        if (select && select.value) {
            usedColors = usedColors.filter(c => c !== select.value);
        }
        document.getElementById(`var-${idx}`).remove();
        updateAllDropdowns();
    }

    function updateAllDropdowns() {
        const allSelects = document.querySelectorAll('.variant-color-select');
        allSelects.forEach(select => {
            const currentValue = select.value;
            let html = '<option value="">Select Color</option>';
            availableColors.forEach(c => {
                if (!usedColors.includes(c.name) || c.name === currentValue) {
                    html += `<option value="${c.name}" ${c.name === currentValue ? 'selected' : ''}>${c.name}</option>`;
                }
            });
            select.innerHTML = html;
        });
    }

    function addSizeRow(idx, name = '', qty = '') {
        const html = `<div class="d-flex gap-2 mb-2 align-items-center"><input type="text" name="variants[${idx}][sizes][name][]" value="${name}" class="clean-input" placeholder="Size" style="width:100px; height:48px;" required autocomplete="off"><input type="number" name="variants[${idx}][sizes][qty][]" value="${qty}" class="clean-input" placeholder="Qty" style="width:100px; height:48px;" required autocomplete="off"><button type="button" class="btn btn-sm text-danger" onclick="this.parentElement.remove()" style="height:48px; width:40px; display:flex; justify-content:center; align-items:center;"><i class="fas fa-times"></i></button></div>`;
        document.getElementById(`sizes-${idx}`).insertAdjacentHTML('beforeend', html);
    }

    function updateDefaultColorDropdown() {
        if (!document.getElementById('default-color-selector')) return;
        const select = document.getElementById('default-color-selector');
        const currentValue = select.value;
        let html = '<option value="">-- Select Default --</option>';
        availableColors.forEach(c => {
            let selected = (c.name === currentValue) ? 'selected' : '';
            html += `<option value="${c.name}" ${selected}>${c.name}</option>`;
        });
        select.innerHTML = html;
    }

    function addSpecRow(key = '', val = '') {
        const html = `<div class="d-flex gap-2 mb-2"><input type="text" name="spec_key[]" value="${key}" class="clean-input" placeholder="Label (e.g. Fabric)" autocomplete="off"><input type="text" name="spec_val[]" value="${val}" class="clean-input" placeholder="Value (e.g. Cottton)" autocomplete="off"><button type="button" class="btn btn-sm text-danger" onclick="this.parentElement.remove()" style="height:48px; width:40px; display:flex; justify-content:center; align-items:center;"><i class="fas fa-times"></i></button></div>`;
        document.getElementById('specs-container').insertAdjacentHTML('beforeend', html);
    }
</script>

<?php include 'includes/admin_footer.php'; ?>