<?php
require_once 'includes/db_config.php';

// If already logged in, redirect to dashboard
if (isset($_SESSION['admin_id'])) {
    header("Location: dashboard.php");
    exit;
}

// Handle login logic
$login_error = '';
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $input_login = sanitize_input($_POST['username'] ?? ''); // Accepts Username OR Email
    $password = $_POST['password'] ?? '';
    
    if (empty($input_login) || empty($password)) {
        $login_error = "Username/Email and password are required.";
    } else {
        // MODIFIED: Check against BOTH username AND email columns
        $stmt = $conn->prepare("SELECT id, username, password FROM admin_users WHERE username = ? OR email = ?");
        // Bind the same input to both placeholders ("ss" means two strings)
        $stmt->bind_param("ss", $input_login, $input_login);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $row = $result->fetch_assoc();
            if (password_verify($password, $row['password'])) {
                $_SESSION['admin_id'] = $row['id'];
                $_SESSION['admin_username'] = $row['username'];
                header("Location: dashboard.php");
                exit;
            } else {
                $login_error = "Invalid credentials.";
            }
        } else {
            $login_error = "Invalid credentials.";
        }
        $stmt->close();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Login - Miss.Sogasu</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin.css"> 
</head>
<body class="login-page-body">
    <div class="login-container">
        <div class="login-header">
            <img src="../assets/img/logo2.png" alt="Miss.Sogasu Logo" style="max-width: 150px; margin-bottom: 15px;">
            <p>Admin Panel</p>
        </div>

        <div class="login-body">
            <?php if ($login_error): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <strong>Error:</strong> <?php echo htmlspecialchars($login_error); ?>
                    <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
            <?php endif; ?>

            <form method="POST" action="" id="loginForm">
                <div class="form-group">
                    <!-- MODIFIED: Updated Label and Placeholder -->
                    <label for="username">Username or Email</label>
                    <input type="text" class="form-control" id="username" name="username" placeholder="Enter username or email" required>
                </div>

                <div class="form-group">
                    <label for="password">Password</label>
                    <div class="input-group">
                        <input type="password" class="form-control" id="password" name="password" placeholder="Enter your password" required>
                        <div class="input-group-append">
                            <button class="btn btn-outline-secondary toggle-password" type="button">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                    </div>
                </div>

                <button type="submit" class="btn-login">
                    <span class="btn-text">Login</span>
                    <span class="login-loading" style="display: none;"><i class="fas fa-spinner fa-spin"></i> Logging in...</span>
                </button>
            </form>

            <div class="login-footer-text">
                <p>&copy; <?php echo date("Y"); ?> Miss.Sogasu. All rights reserved.</p>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // 1. Button Loading State
        document.getElementById('loginForm').addEventListener('submit', function() {
            const loginButton = this.querySelector('.btn-login');
            loginButton.disabled = true;
            loginButton.querySelector('.btn-text').style.display = 'none';
            loginButton.querySelector('.login-loading').style.display = 'inline-block';
        });

        // 2. Password Visibility Toggle
        document.querySelector('.toggle-password').addEventListener('click', function() {
            const input = document.getElementById('password');
            const icon = this.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        });
    </script>
</body>
</html>