<?php
require_once 'includes/db_config.php';
check_login();

// --- 1. KEY METRICS ---
// Revenue
$rev_res = $conn->query("SELECT SUM(total_amount) as total FROM orders WHERE payment_status = 'Completed'");
$total_revenue = $rev_res->fetch_assoc()['total'] ?? 0;

// Total Orders
$ord_res = $conn->query("SELECT COUNT(*) as total FROM orders");
$total_orders = $ord_res->fetch_assoc()['total'] ?? 0;

// Total Customers
$cust_res = $conn->query("SELECT COUNT(*) as total FROM users");
$total_customers = $cust_res->fetch_assoc()['total'] ?? 0;

// Active Products
$prod_res = $conn->query("SELECT COUNT(*) as total FROM products WHERE is_active = 1");
$total_products = $prod_res->fetch_assoc()['total'] ?? 0;

// --- 2. DATA FOR CHARTS ---
// A. Monthly Revenue (Last 6 Months)
$months = [];
$revenues = [];
for ($i = 5; $i >= 0; $i--) {
    $month_start = date("Y-m-01", strtotime("-$i months"));
    $month_end = date("Y-m-t", strtotime("-$i months"));
    $month_label = date("M Y", strtotime("-$i months"));

    $sql = "SELECT SUM(total_amount) as total FROM orders WHERE payment_status='Completed' AND created_at BETWEEN '$month_start 00:00:00' AND '$month_end 23:59:59'";
    $res = $conn->query($sql);
    $row = $res->fetch_assoc();

    $months[] = $month_label;
    $revenues[] = $row['total'] ?? 0;
}

// B. Order Status Breakdown
$status_sql = "SELECT order_status, COUNT(*) as count FROM orders GROUP BY order_status";
$status_res = $conn->query($status_sql);
$status_labels = [];
$status_data = [];
$status_colors = [];

while ($row = $status_res->fetch_assoc()) {
    $status_labels[] = $row['order_status'];
    $status_data[] = $row['count'];
    // Assign colors based on status
    if ($row['order_status'] == 'Completed' || $row['order_status'] == 'Delivered') $status_colors[] = '#28a745'; // Green
    elseif ($row['order_status'] == 'Pending') $status_colors[] = '#ffc107'; // Yellow
    elseif ($row['order_status'] == 'Processing') $status_colors[] = '#17a2b8'; // Blue
    elseif ($row['order_status'] == 'Cancelled') $status_colors[] = '#dc3545'; // Red
    else $status_colors[] = '#6c757d'; // Grey
}

// --- 3. TABLES DATA ---
// Top Products
$top_products = $conn->query("SELECT p.name, p.price, SUM(oi.quantity) as total_sold, (SELECT image_path FROM product_images pi WHERE pi.product_id = p.id AND pi.is_primary = 1 LIMIT 1) as image FROM order_items oi JOIN products p ON oi.product_id = p.id GROUP BY p.id ORDER BY total_sold DESC LIMIT 5");

// Recent Orders
$recent_orders = $conn->query("SELECT o.*, u.full_name as reg_name FROM orders o LEFT JOIN users u ON o.user_id = u.id ORDER BY o.created_at DESC LIMIT 6");

// --- 4. INVENTORY RISK: ALL OUT OF STOCK PRODUCTS ---
$out_of_stock_sql = "SELECT ps.*, p.name, p.slug,
                    (SELECT image_path FROM product_images pi WHERE pi.product_id = p.id AND pi.is_primary = 1 LIMIT 1) as image
                    FROM product_sizes ps
                    JOIN products p ON ps.product_id = p.id
                    WHERE ps.stock_quantity <= 0
                    ORDER BY p.name ASC";
$out_of_stock_res = $conn->query($out_of_stock_sql);


$page_title = "Executive Dashboard";
include 'includes/admin_header.php';
?>

<div class="container-fluid">

    <!-- 1. KEY METRICS ROW -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="stat-card h-100 border-left-primary">
                <div class="stat-icon green"><i class="fas fa-rupee-sign"></i></div>
                <div class="stat-content">
                    <h5>Total Revenue</h5>
                    <p class="mb-0">₹<?= number_format($total_revenue, 2) ?></p>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="stat-card h-100 border-left-success">
                <div class="stat-icon blue"><i class="fas fa-shopping-bag"></i></div>
                <div class="stat-content">
                    <h5>Total Orders</h5>
                    <p class="mb-0"><?= number_format($total_orders) ?></p>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="stat-card h-100 border-left-info">
                <div class="stat-icon purple"><i class="fas fa-users"></i></div>
                <div class="stat-content">
                    <h5>Total Customers</h5>
                    <p class="mb-0"><?= number_format($total_customers) ?></p>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="stat-card h-100 border-left-warning">
                <div class="stat-icon orange"><i class="fas fa-tshirt"></i></div>
                <div class="stat-content">
                    <h5>Active Products</h5>
                    <p class="mb-0"><?= number_format($total_products) ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- 2. CHARTS SECTION -->
    <div class="row mb-4">
        <!-- Area Chart: Monthly Revenue -->
        <div class="col-xl-8 col-lg-7">
            <div class="card shadow mb-4 h-100">
                <div class="card-header card-header-custom py-3 d-flex flex-row align-items-center justify-content-between">
                    <h5 class="m-0 font-weight-bold">Sales Overview (Last 6 Months)</h5>
                </div>
                <div class="card-body">
                    <div class="chart-area" style="height: 320px;">
                        <canvas id="myAreaChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Pie Chart: Order Status -->
        <div class="col-xl-4 col-lg-5">
            <div class="card shadow mb-4 h-100">
                <div class="card-header card-header-custom py-3">
                    <h5 class="m-0 font-weight-bold">Order Status Distribution</h5>
                </div>
                <div class="card-body">
                    <div class="chart-pie" style="height: 320px; position: relative;">
                        <canvas id="myPieChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- 3. DETAILED DATA ROW -->
    <div class="row">
        <!-- Left: Recent Orders -->
        <div class="col-lg-8 mb-4">
            <div class="card h-100 shadow-sm">
                <div class="card-header card-header-custom d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Recent Order Feed</h5>
                    <a href="orders.php" class="btn btn-sm btn-light text-dark font-weight-bold">View All</a>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover align-middle mb-0">
                            <thead class="bg-light">
                                <tr>
                                    <th class="border-top-0">Order ID</th>
                                    <th class="border-top-0">Customer</th>
                                    <th class="border-top-0">Status</th>
                                    <th class="border-top-0">Amount</th>
                                    <th class="border-top-0 text-right">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($recent_orders->num_rows > 0): ?>
                                    <?php while ($o = $recent_orders->fetch_assoc()):
                                        $ship = json_decode($o['shipping_address_details'], true);
                                        $name = $ship['full_name'] ?? ($o['reg_name'] ?? 'Guest');

                                        $s_bg = 'secondary';
                                        if ($o['order_status'] == 'Pending') $s_bg = 'warning';
                                        elseif ($o['order_status'] == 'Processing') $s_bg = 'info';
                                        elseif ($o['order_status'] == 'Shipped') $s_bg = 'primary';
                                        elseif ($o['order_status'] == 'Delivered') $s_bg = 'success';
                                    ?>
                                        <tr>
                                            <td><strong>#<?= str_pad($o['id'], 6, '0', STR_PAD_LEFT) ?></strong></td>
                                            <td><?= htmlspecialchars($name) ?></td>
                                            <td><span class="badge badge-<?= $s_bg ?>"><?= $o['order_status'] ?></span></td>
                                            <td><strong>₹<?= number_format($o['total_amount'], 2) ?></strong></td>
                                            <td class="text-right"><a href="order-details.php?id=<?= $o['id'] ?>" class="btn btn-sm btn-outline-dark"><i class="fas fa-arrow-right"></i></a></td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="text-center py-5">No recent orders.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Right: Top Products -->
        <div class="col-lg-4 mb-4">
            <div class="card mb-4 shadow-sm h-100">
                <div class="card-header card-header-custom">
                    <h5 class="mb-0"><i class="fas fa-trophy mr-2 text-warning"></i> Best Sellers</h5>
                </div>
                <div class="card-body p-0">
                    <?php if ($top_products->num_rows > 0): ?>
                        <ul class="list-group list-group-flush">
                            <?php while ($prod = $top_products->fetch_assoc()): ?>
                                <li class="list-group-item d-flex align-items-center justify-content-between p-3">
                                    <div class="d-flex align-items-center">
                                        <img src="../<?= $prod['image'] ?? 'assets/img/placeholder.png' ?>" class="rounded mr-3 border" style="width: 45px; height: 45px; object-fit: cover;">
                                        <div>
                                            <span class="d-block font-weight-bold text-dark"><?= htmlspecialchars($prod['name']) ?></span>
                                            <small class="text-muted">₹<?= number_format($prod['price']) ?></small>
                                        </div>
                                    </div>
                                    <span class="badge badge-success badge-pill"><?= $prod['total_sold'] ?> Sold</span>
                                </li>
                            <?php endwhile; ?>
                        </ul>
                    <?php else: ?>
                        <div class="p-4 text-center text-muted">No sales data yet.</div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- 4. OUT OF STOCK PRODUCTS ROW -->
    <div class="row mt-4">
        <div class="col-12">
            <div class="card shadow-sm">
                <div class="card-header card-header-custom bg-dark d-flex justify-content-between align-items-center">
                    <h5 class="mb-0 text-white"><i class="fas fa-exclamation-triangle mr-2 text-danger"></i> Out of Stock Variations</h5>
                    <span class="badge badge-danger"><?= $out_of_stock_res->num_rows ?> Items Empty</span>
                </div>
                <div class="card-body card-body-no-padding">
                    <div class="table-responsive">
                        <table class="table table-hover align-middle mb-0">
                            <thead class="bg-light">
                                <tr>
                                    <th>Image</th>
                                    <th>Product Name</th>
                                    <th>Color Group</th>
                                    <th>Size</th>
                                    <th>Current Stock</th>
                                    <th class="text-right">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($out_of_stock_res->num_rows > 0): ?>
                                    <?php while ($oos = $out_of_stock_res->fetch_assoc()): ?>
                                        <tr>
                                            <td>
                                                <img src="../<?= $oos['image'] ?? 'assets/img/placeholder.png' ?>" 
                                                     class="rounded border" 
                                                     style="width: 45px; height: 45px; object-fit: cover;">
                                            </td>
                                            <td>
                                                <span class="font-weight-bold text-dark"><?= htmlspecialchars($oos['name']) ?></span>
                                            </td>
                                            <td><?= htmlspecialchars($oos['color_group']) ?></td>
                                            <td><span class="badge badge-secondary"><?= htmlspecialchars($oos['size']) ?></span></td>
                                            <td><span class="text-danger font-weight-bold">SOLD OUT</span></td>
                                            <td class="text-right">
                                                <a href="product-edit.php?id=<?= $oos['product_id'] ?>" class="btn btn-sm btn-primary">
                                                    <i class="fas fa-plus mr-1"></i> Restock
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="6" class="text-center py-5 text-success">
                                            <i class="fas fa-check-circle fa-2x mb-2"></i><br>
                                            Every product variant is currently in stock!
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>


<!-- SCRIPTS FOR CHARTS -->
<script>
    // Wait for the DOM to be fully loaded before running this script
    document.addEventListener("DOMContentLoaded", function() {

        // --- 1. AREA CHART (REVENUE) ---
        // Interactive: Grows smoothly from left to right
        var ctx = document.getElementById("myAreaChart");
        if (ctx) {
            var myLineChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: <?= json_encode($months) ?>,
                    datasets: [{
                        label: "Earnings",
                        lineTension: 0.4, // Smoother curves
                        backgroundColor: "rgba(218, 1, 3, 0.05)",
                        borderColor: "#da0103",
                        pointRadius: 4,
                        pointBackgroundColor: "#fff",
                        pointBorderColor: "#da0103",
                        pointHoverRadius: 6, // Bigger hover effect
                        pointHoverBackgroundColor: "#da0103",
                        pointHoverBorderColor: "#fff",
                        pointBorderWidth: 2,
                        data: <?= json_encode($revenues) ?>,
                        fill: true
                    }],
                },
                options: {
                    maintainAspectRatio: false,
                    responsive: true,
                    animation: {
                        duration: 2000, // 2 Seconds animation
                        easing: 'easeOutQuart' // Smooth deceleration
                    },
                    interaction: {
                        mode: 'index',
                        intersect: false, // Tooltip appears when hovering the x-axis area, not just the dot
                    },
                    scales: {
                        x: { 
                            grid: { display: false, drawBorder: false },
                            ticks: { color: '#858796' }
                        },
                        y: { 
                            ticks: { 
                                color: '#858796',
                                callback: function(value) { return '₹' + value; } 
                            },
                            grid: { 
                                color: "rgb(234, 236, 244)", 
                                borderDash: [2], 
                                drawBorder: false 
                            } 
                        },
                    },
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            backgroundColor: "rgb(255,255,255)",
                            bodyColor: "#858796",
                            titleMarginBottom: 10,
                            titleColor: '#6e707e',
                            titleFont: { size: 14 },
                            borderColor: '#dddfeb',
                            borderWidth: 1,
                            padding: 15,
                            displayColors: false,
                            callbacks: {
                                label: function(context) { 
                                    return ' Revenue: ₹' + context.parsed.y.toLocaleString(); 
                                }
                            }
                        }
                    }
                }
            });
        }

        // 2. PIE CHART (ORDER STATUS)
        var ctxPie = document.getElementById("myPieChart");
        if (ctxPie) {
            var myPieChart = new Chart(ctxPie, {
                type: 'doughnut',
                data: {
                    labels: <?= json_encode($status_labels) ?>,
                    datasets: [{
                        data: <?= json_encode($status_data) ?>,
                        backgroundColor: <?= json_encode($status_colors) ?>,
                        hoverBorderColor: "rgba(234, 236, 244, 1)",
                    }],
                },
                options: {
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom'
                        }
                    },
                    cutout: '70%',
                },
            });
        }
    });
</script>

<?php include 'includes/admin_footer.php'; ?>